/*
 * Decompiled with CFR 0.152.
 */
package org.dcm4chee.arc.storage.filesystem;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.nio.file.DirectoryNotEmptyException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileStore;
import java.nio.file.FileSystemException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.concurrent.ThreadLocalRandom;
import org.dcm4chee.arc.conf.StorageDescriptor;
import org.dcm4chee.arc.metrics.MetricsService;
import org.dcm4chee.arc.storage.AbstractStorage;
import org.dcm4chee.arc.storage.ReadContext;
import org.dcm4chee.arc.storage.WriteContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemStorage
extends AbstractStorage {
    private static final Logger LOG = LoggerFactory.getLogger(FileSystemStorage.class);
    private static final int COPY_BUFFER_SIZE = 8192;
    private final URI rootURI;
    private final Path checkMountFilePath;
    private final boolean noopOnFileExists;
    private final boolean randomPathOnFileExists;
    private final OpenOption[] openOptions;
    private final CreateDirectories createDirectories;
    private final int retryCreateDirectories;

    public FileSystemStorage(StorageDescriptor descriptor, MetricsService metricsService) {
        super(descriptor, metricsService);
        OpenOption[] openOptionArray;
        this.rootURI = this.ensureTrailingSlash(descriptor.getStorageURI());
        String checkMountFile = descriptor.getProperty("checkMountFile", null);
        this.checkMountFilePath = checkMountFile != null ? Paths.get(this.rootURI.resolve(checkMountFile)) : null;
        String onFileExists = descriptor.getProperty("onFileExists", null);
        this.randomPathOnFileExists = onFileExists == null;
        this.noopOnFileExists = "NOOP".equalsIgnoreCase(onFileExists);
        String fileOpenOption = descriptor.getProperty("fileOpenOption", null);
        if (fileOpenOption != null) {
            OpenOption[] openOptionArray2 = new OpenOption[2];
            openOptionArray2[0] = StandardOpenOption.CREATE_NEW;
            openOptionArray = openOptionArray2;
            openOptionArray2[1] = StandardOpenOption.valueOf(fileOpenOption);
        } else {
            OpenOption[] openOptionArray3 = new OpenOption[1];
            openOptionArray = openOptionArray3;
            openOptionArray3[0] = StandardOpenOption.CREATE_NEW;
        }
        this.openOptions = openOptionArray;
        this.createDirectories = Boolean.parseBoolean(descriptor.getProperty("altCreateDirectories", null)) ? FileSystemStorage::altCreateDirectories : Files::createDirectories;
        this.retryCreateDirectories = Integer.parseInt(descriptor.getProperty("retryCreateDirectories", "0"));
    }

    protected Logger log() {
        return LOG;
    }

    private static Path altCreateDirectories(Path path, FileAttribute<?> ... fileAttributes) throws IOException {
        try {
            return Files.createDirectory(path, fileAttributes);
        }
        catch (FileAlreadyExistsException e) {
            return path;
        }
        catch (NoSuchFileException e) {
            FileSystemStorage.createParentDirectories(path, fileAttributes);
            return Files.createDirectory(path, fileAttributes);
        }
    }

    private static void createParentDirectories(Path path, FileAttribute<?> ... fileAttributes) throws IOException {
        Path parent = path.getParent();
        if (parent == null) {
            throw new FileSystemException(path.toString(), null, "Unable to determine if root directory exists");
        }
        try {
            Files.createDirectory(parent, fileAttributes);
        }
        catch (NoSuchFileException e) {
            FileSystemStorage.createParentDirectories(parent, fileAttributes);
            Files.createDirectory(parent, fileAttributes);
        }
    }

    private URI ensureTrailingSlash(URI uri) {
        String s = uri.toString();
        return s.charAt(s.length() - 1) == '/' ? uri : URI.create(s + "/");
    }

    public boolean isAccessable() {
        return this.checkMountFilePath == null || Files.notExists(this.checkMountFilePath, new LinkOption[0]);
    }

    public boolean exists(ReadContext ctx) {
        Path path = Paths.get(this.rootURI.resolve(ctx.getStoragePath()));
        return Files.exists(path, new LinkOption[0]);
    }

    public long getContentLength(ReadContext ctx) throws IOException {
        Path path = Paths.get(this.rootURI.resolve(ctx.getStoragePath()));
        return Files.size(path);
    }

    public long getUsableSpace() throws IOException {
        return this.getFileStore().getUsableSpace();
    }

    private FileStore getFileStore() throws IOException {
        Path dir = Paths.get(this.rootURI);
        this.createDirectories(dir);
        return Files.getFileStore(dir);
    }

    private Path createDirectories(Path path) throws IOException {
        int retries = this.retryCreateDirectories;
        while (true) {
            try {
                return this.createDirectories.apply(path, new FileAttribute[0]);
            }
            catch (NoSuchFileException e) {
                if (--retries < 0) {
                    throw e;
                }
                LOG.info("Failed to create directories {} - retry:\n", (Object)path, (Object)e);
                continue;
            }
            break;
        }
    }

    public long getTotalSpace() throws IOException {
        return this.getFileStore().getTotalSpace();
    }

    protected OutputStream openOutputStreamA(WriteContext ctx) throws IOException {
        Path path = Paths.get(this.rootURI.resolve(ctx.getStoragePath()));
        Path dir = path.getParent();
        this.createDirectories(dir);
        Object stream = null;
        while (true) {
            try {
                ctx.setStoragePath(this.rootURI.relativize(path.toUri()).toString());
                return Files.newOutputStream(path, this.openOptions);
            }
            catch (FileAlreadyExistsException e) {
                if (this.noopOnFileExists) {
                    ctx.setDeletionLock(true);
                    return OutputStream.nullOutputStream();
                }
                if (this.randomPathOnFileExists) {
                    path = dir.resolve(String.format("%08X", ThreadLocalRandom.current().nextInt()));
                    continue;
                }
                ctx.setDeletionLock(true);
                throw e;
            }
            break;
        }
    }

    protected void copyA(InputStream in, WriteContext ctx) throws IOException {
        try (OutputStream out = this.openOutputStreamA(ctx);){
            byte[] b = new byte[8192];
            int read = in.read(b, 0, 8192);
            if (read <= 0) {
                throw new IOException("No bytes to copy");
            }
            do {
                out.write(b, 0, read);
            } while ((read = in.read(b, 0, 8192)) > 0);
        }
    }

    protected InputStream openInputStreamA(ReadContext ctx) throws IOException {
        Path path = Paths.get(this.rootURI.resolve(ctx.getStoragePath()));
        return Files.newInputStream(path, new OpenOption[0]);
    }

    protected void deleteObjectA(String storagePath) throws IOException {
        Path path = Paths.get(this.rootURI.resolve(storagePath));
        Files.delete(path);
        this.deleteEmptyDirectories(path);
    }

    private void deleteEmptyDirectories(Path path) {
        Path rootPath = Paths.get(this.rootURI);
        Path dirPath = path.getParent();
        while (!dirPath.equals(rootPath)) {
            try {
                Files.deleteIfExists(dirPath);
            }
            catch (DirectoryNotEmptyException ignore) {
                break;
            }
            catch (IOException e) {
                LOG.warn("Failed to delete directory {}", (Object)path, (Object)e);
                break;
            }
            dirPath = dirPath.getParent();
        }
    }

    public String toString() {
        return "FileSystemStorage{" + this.rootURI + "}";
    }

    @FunctionalInterface
    private static interface CreateDirectories {
        public Path apply(Path var1, FileAttribute<?> ... var2) throws IOException;
    }
}

